/* list a psf font, to see what the mapping is (if present).
 * Works on amd64, ppc, ppc64 so I guess it is endian-safe.
 * Output to stderr is very noisy, particuarly for type 2, but then it has
 * only been tested for 2 and 3 byte UTF-8 sequences so far, so it could
 * still fail.
 * © 2007 Ken Moffat <ken at linuxfromscratch dot org>
 * LICENSE: GPL v2 or, at your choice, any later version.
 */ 

#include <stdio.h>
#include <stdlib.h>
#define _GNU_SOURCE /* we want the gnu version of basename */
#include <string.h>
#include <utils.h>
/* for int32_t le32_to_cpu(int32_t x)
 * and le16_to_cpu */
#include <sys/types.h>

FILE *fp;
unsigned char first4[4];
/* type 2 header has an extra 7 le32 fields */
unsigned char type2[28];
unsigned char *p;
unsigned char mode, height;
unsigned char last; /* for type2, which holds UTF-8 separated by xff, last byte we have read*/
int bytes_per_char;
int fontsize, headersize;
int psftype=0;
int table;
unsigned char *filename, *fullname;
unsigned char *prog;

/* from psf.h in kbd */
#define PSF1_MAGIC0     0x36
#define PSF1_MAGIC1     0x04

#define PSF1_MODE512    0x01
#define PSF1_MODEHASTAB 0x02
#define PSF1_MODEHASSEQ 0x04
#define PSF1_MAXMODE    0x05

#define PSF2_MAGIC0     0x72
#define PSF2_MAGIC1     0xb5
#define PSF2_MAGIC2     0x4a
#define PSF2_MAGIC3     0x86

unsigned int decode(unsigned char first, int num){
/* for ascii (num==1) c is already the value, just need to copy it to u.
 * otherwise use an array of 6 chars, put c at [0] and read the rest
 * for each of the rest, if it doesn't start 10 it is bad UTF-8, so range is
 * 80 to bf.  Then, decode it.
 * 
 */

 unsigned char bytes[6],c;
 unsigned int decoded;
 int i,j;
 	if (num==1) {
		decoded=first;
		return decoded;
	}
	bytes[0]=c;
	for (j=1;j<num;j++) {
		/* read one byte, and validate it as a trailing byte */
		i=fread(&bytes[j], 1, 1, fp);
		if(i!=1) {
			fprintf(stderr, "file i/o error (truncated table) at %#04x\n", table);
			exit(1);
		}
		last=bytes[j];
		if ( bytes[j]<0x80 ) {
			fprintf(stderr, "invalid UTF-8 trailing byte %02x at sequence %d\n",
				bytes[j], j);
			exit(1);
		}
		if ( bytes[j]>0xbf ) {
			fprintf(stderr, "invalid UTF-8 trailing byte %02x at sequence %d\n",
				bytes[j], j);
			exit(1);
		}
		fprintf(stderr,"successful read of %02x at sequence%d\n", bytes[j], j);
	}
	/* ok, we have the unicode in bytes, need to decode it.
	 * assume the result will fit into 32 bits because fe and ff are special
	 */
	/* we already analysed the first byte, to get num:
	 * two 110..... c0-df
	 * three 1110.... e0-ef
	 * four 11110... f0-f7
	 * five 111110.. f8-fb
	 * six 1111110. fc-fd
	 */
	if (first<0xe0) decoded=(first & 0x1f);
	else if(first<0xf0) decoded=(first & 0x0f);
	else if(first<0xf8) decoded=(first & 0x07);
	else if(first<0xfc) decoded=(first & 0x03);
	else decoded=(first & 0x01);

	/* for remaining bytes, shift decoded left by 6,
	 * and the byte with 0x3f to drop initial 'b10',
	 * add the result to decoded.
	 */
	for(j=1;j<num;j++) {
		fprintf(stderr,"decoded was %04x and byte is %02x\n",decoded,c);
		decoded=(decoded <<6);
		c=(bytes[j] & 0x3f);
		decoded=decoded+c;
		fprintf(stderr,"decoded now %04x\n",decoded);
	}
	return decoded;
	
}

int openpsf(char *filename) {
 int bytes;
 int i, *iptr;
 unsigned char c; int count;
	/* open it here for reading */
	fprintf(stderr,"Trying to open %s\n", fullname);
	fp=fopen(fullname, "r");
	if (fp == NULL) {
		fprintf(stderr, "file open failed for %s\n", fullname);
		exit(1);
	}
	fprintf(stderr, "it exists\n");
	/* now check that it really is a psf, and set the details */
	bytes=fread(first4, 1, 4, fp);
	if (bytes != 4) {
		fprintf(stderr, "failed to read first 4 bytes of header\n");
		exit(1);
	}
	if( (first4[0]==PSF1_MAGIC0) && (first4[1]==PSF1_MAGIC1) ) {
		fprintf(stderr,"it is type 1\n");
		psftype=1;
		mode=first4[2];
		height=first4[3];
		if(mode > 5) {
			fprintf(stderr,"psf file with mode %02x is unsupported\n");
			exit(1);
		}
		if (! (mode & 0x02)) {
			fprintf(stderr,"psf1 file with mode %02x - no unicode table\n");
			exit(1);
		}
		if (mode & 0x01) fontsize=512;
		else fontsize=256;
		fprintf(stderr,"font size is %d from mode %02x\n", fontsize, mode);
		bytes_per_char=height;
		fprintf(stderr,"bytes per character is %d\n", bytes_per_char);
		headersize=4;
		return psftype;
	}
	if( (first4[0]==PSF2_MAGIC0) && (first4[1]==PSF2_MAGIC1)
	 && (first4[2]==PSF2_MAGIC2) && (first4[3]==PSF2_MAGIC3) ) {
		/* read the extended header */
		bytes=fread(type2, 1, 28, fp);
		if (bytes != 28) {
			fprintf(stderr, "failed to read next 28 bytes of header\n");
			exit(1);
		}
		iptr=&type2[4];
		headersize=le32_to_cpu(*iptr);
		fprintf(stderr,"header size is %d\n", headersize);

		iptr=&type2[8];
		if(! (le32_to_cpu(*iptr) & 0x01)) {
			fprintf(stderr, "file mode is %02x, no unicode table\n",
				le32_to_cpu(*iptr));
		}
		iptr=&type2[12];
		fontsize=le32_to_cpu(*iptr);
		fprintf(stderr, "file contains %d glyphs\n", fontsize);
		iptr=&type2[16];
		bytes_per_char=le32_to_cpu(*iptr);
		fprintf(stderr, "bytes per character is %d\n", bytes_per_char);
		psftype=2;
		return psftype;
	}
	fprintf(stderr,"%s is not a psf file\n",fullname);
	exit(1);
}

void usage() {
	fprintf(stderr,"%s : list the contents of a unicode psf file\n", prog);
	fprintf(stderr,"Usage: %s /path/to/file \n", prog);
	exit(1);
}

void validate(int argc, char *argv[]) {
 char *f, *p, *dirc, *basec, *dname, *bname;
	p=argv[0];
	fprintf(stderr, "running %s\n", p);
	dirc=strdup(p);
	basec=strdup(p);
	dname=dirname(dirc);
	bname=basename(basec);
	prog=bname;
	if (argc != 2) {
		usage();
	}
	f=argv[1];
	basec=strdup(f);
	bname=basename(basec);
	filename=bname;
	fullname=f;
}

process_type1(int glyphnum) {
 /* read values, if any, for one position */
 u_int16_t i,u,r;
	fprintf(stdout, "%#04x", glyphnum);
	do {
		i=fread(&r, 2, 1, fp);
		if(i!=1) {
			fprintf(stderr, "file i/o error (truncated table) at %#04x\n", table);
			exit(1);
		}
		table+=2;
		u=le16_to_cpu(r);
		if(u!=0xFFFF) {
			fprintf(stdout,"\tU+%04x",u);
		}
	} while (u!=0xFFFF);
	fprintf(stdout, "\n");
}

process_type2(int glyphnum) {
/* read values, if any, for one position:
 * type 2 is composed of UTF-8, so detect xff, xfe, ascii
 * or otherwise determine how many bytes are needed.
 */
 unsigned char c,k; int i,j; unsigned int u; int cnt=0;
	fprintf(stdout, "%#04x", glyphnum);
	do {
		i=fread(&c, 1, 1, fp);
		if(i!=1) {
			fprintf(stderr, "file i/o error (truncated table) at %#04x\n", table);
			exit(1);
		}
		last=c;
		table++;
		if(c==0xfe) {
			fprintf(stderr,"Found combining sequence in type2\n");
			exit(3);
		}
		if (c!=0xff) {
			/* determine how many bytes in this UTF-8 character  */
			/* two 110..... c0-df
			 * three 1110.... e0-ef
			 * four 11110... f0-f7
			 * five 111110.. f8-fb
			 * six 1111110. fc-fd
			 * fe already assumed to start a combining sequence */
			j=0;
			if(c>0xfb) j=6;
			else if(c>0xf7) j=5;
			else if(c>0xef) j=4;
			else if(c>0xdf) j=3;
			else if(c>0xbf) j=2;
			else if(c<0x80) j=1;
			if (j==0) {
				fprintf(stderr,"invalid UTF-8 byte %x\n",c);
				exit(2);
			}
			fprintf(stderr,"decided that UTF-8 beginning %02x will be %d bytes\n", c,j);
			u=decode(c,j);
			fprintf(stdout,"\tU+%04x",u);
			cnt++;
		} else {
			c=u;
			fprintf(stdout,"\n");
		}
	} while (last != 0xff);
	fprintf(stderr,"found %d mappings at position %02x\n", cnt, glyphnum);
}

main(int argc, char *argv[])
{
 int i,k;
	validate(argc, argv);
	psftype=openpsf(fullname);
	if((psftype <1) || (psftype>2)) {
		fprintf(stderr, "Internal error, openpsf returned type %d\n", psftype);
		exit(1);
	}
	fprintf(stderr,"file opened, psftype is %d\n", psftype);

	/* calculate where the table starts */
	table=headersize+(fontsize*bytes_per_char);
	fprintf(stderr,"table starts at offset %#04x\n",table);

	/* position for reading at ts */
	i=fseek(fp, table, SEEK_SET);
	if(i!=0) {
		fprintf(stderr, "Unable to seek to %#04x\n", table);
		exit(1);
	}

	/* main loop to read the table - each pass is a new position
	 * any failure to read means table is short
	 */
	fprintf(stdout, "Character mapping for type %d psf file %s\n\n", psftype, filename);
	for(k=0;k<fontsize;k++) {
		if(psftype==1) process_type1(k);
		else process_type2(k);	
	}
	exit(0);
}

